<?php

namespace App\Http\Controllers\Store;

use App\Http\Controllers\Controller;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Barryvdh\DomPDF\Facade\Pdf;

class PurchaseHistoryController extends Controller
{
    public function index(Request $request)
    {
        $user = Auth::user();
        
        // Get filter parameters
        $status = $request->get('status', 'all');
        $package_type = $request->get('package_type', 'all');
        $date_range = $request->get('date_range', 'all');
        $search = $request->get('search');
        
        // Base query
        $query = Transaction::with(['package', 'server', 'coupon'])
            ->where('user_id', $user->id)
            ->orderBy('created_at', 'desc');
        
        // Apply filters
        if ($status !== 'all') {
            $query->where('status', $status);
        }
        
        if ($package_type !== 'all') {
            $query->whereHas('package', function($q) use ($package_type) {
                $q->where('type', $package_type);
            });
        }
        
        if ($search) {
            $query->where(function($q) use ($search) {
                $q->where('transaction_id', 'LIKE', "%{$search}%")
                  ->orWhereHas('package', function($subQ) use ($search) {
                      $subQ->where('name', 'LIKE', "%{$search}%");
                  });
            });
        }
        
        // Apply date range filter
        switch ($date_range) {
            case '7_days':
                $query->where('created_at', '>=', now()->subDays(7));
                break;
            case '30_days':
                $query->where('created_at', '>=', now()->subDays(30));
                break;
            case '90_days':
                $query->where('created_at', '>=', now()->subDays(90));
                break;
            case '1_year':
                $query->where('created_at', '>=', now()->subYear());
                break;
        }
        
        // Paginate results
        $transactions = $query->paginate(15)->withQueryString();
        
        // Get statistics
        $stats = $this->getUserStats($user);
        
        return view('store.purchase-history', compact('transactions', 'stats', 'status', 'package_type', 'date_range', 'search'));
    }
    
    public function show($transactionId)
    {
        $transaction = Transaction::with(['package', 'server', 'coupon', 'user'])
            ->where('user_id', Auth::id())
            ->findOrFail($transactionId);
            
        return view('store.purchase-detail', compact('transaction'));
    }
    
    public function downloadReceipt($transactionId)
    {
        $transaction = Transaction::with(['package', 'server', 'coupon', 'user'])
            ->where('user_id', Auth::id())
            ->where('status', 'completed')
            ->findOrFail($transactionId);
        
        $data = [
            'transaction' => $transaction,
            'company' => [
                'name' => config_value('site_name'),
                'website' => config_value('site_url'),
            ]
        ];
        
        $pdf = Pdf::loadView('store.receipt', $data);
        $pdf->setPaper('a4', 'portrait');
        
        $filename = 'receipt-' . $transaction->id . '-' . date('Y-m-d') . '.pdf';
        
        return $pdf->download($filename);
    }
    
    private function getUserStats($user)
    {
        $baseQuery = Transaction::where('user_id', $user->id);
        
        return [
            'total_purchases' => (clone $baseQuery)->where('status', 'completed')->count(),
            'total_spent' => (clone $baseQuery)->where('status', 'completed')->sum('amount'),
            'pending_transactions' => (clone $baseQuery)->where('status', 'pending')->count(),
            'recent_purchase' => (clone $baseQuery)->where('status', 'completed')->latest()->first(),
            'most_purchased_type' => $this->getMostPurchasedType($user),
            'monthly_spending' => $this->getMonthlySpending($user),
        ];
    }
    
    private function getMostPurchasedType($user)
    {
        return Transaction::where('user_id', $user->id)
            ->where('status', 'completed')
            ->join('packages', 'transactions.package_id', '=', 'packages.id')
            ->selectRaw('packages.type, COUNT(*) as count')
            ->groupBy('packages.type')
            ->orderBy('count', 'desc')
            ->first();
    }
    
    private function getMonthlySpending($user)
    {
        return Transaction::where('user_id', $user->id)
            ->where('status', 'completed')
            ->where('created_at', '>=', now()->subMonths(6))
            ->selectRaw('DATE_FORMAT(created_at, "%Y-%m") as month, SUM(amount) as total')
            ->groupBy('month')
            ->orderBy('month')
            ->get();
    }
}